#define __TEXTUREWERKE_C__

//
// Lowpass filter for Gimp
//
// Copyright (C) Lauris Kaplinski 2007-2011
//
// Licensed under GNU GPL
//

static const int debug = 1;

#include <stdio.h>
#ifndef WIN32
#include <unistd.h>
#endif
#include <stdlib.h>
#include <time.h>
#include <math.h>
#include <string.h>

#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>

#ifndef _
#include <libintl.h>
#define _(msgid)	gettext (msgid)
#endif

#include "texturewerke.h"

// Talking with GIMP
static void query ();
static void run (const gchar *name, gint nparams, const GimpParam *param, gint *nreturn_vals, GimpParam **return_vals);

// Name for PDB
#define PLUGIN_NAME "plug-in-texturewerke"

// Parameters
enum {
	// Standard GIMP parameters
	PARAM_RUNMODE, PARAM_IMAGE, PARAM_DRAWABLE,
	// Method
	PARAM_METHOD,
	// Other layers
	PARAM_OTHER_LAYER,
	// Blur colors
	PARAM_COLOR_USE_BLUR,
	// Blur/lowpass
	PARAM_XSIGMA, PARAM_YSIGMA,
	// Polynome parameters
	PARAM_KPOWER,
	PARAM_DEGREE_X, PARAM_DEGREE_Y, PARAM_NSAMPLES, PARAM_COPYLF
};

static GimpPlugInInfo PLUG_IN_INFO = {
	// init_proc
	NULL,
	// quit_proc
	NULL,
	// query_proc
	query,
	// run_proc
	run,
};

/* Macro to define the usual plugin main function */
MAIN ()

/* Add capabilities of this plugin to Procedural DataBase */
static void
query () {
	static GimpParamDef args_in[] = {
		{ GIMP_PDB_INT32, "run_mode", "Interactive, non-interactive" },
		{ GIMP_PDB_IMAGE, "image", "Input image (unused)" },
		{ GIMP_PDB_DRAWABLE, "drawable", "Input drawable" },

		{ GIMP_PDB_INT32, "method", "Operation type (0 - color correction, 1,2 lowpass filter)" },
		{ GIMP_PDB_DRAWABLE, "template", "Template layer" },
		{ GIMP_PDB_INT32, "color_blur", "Blur image and template before adjustment" },
		{ GIMP_PDB_FLOAT, "xsigma", "The width (X sigma) of blur kernel" },
		{ GIMP_PDB_FLOAT, "ysigma", "The height (Y sigma) of blur kernel" },
		{ GIMP_PDB_FLOAT, "kpower", "The power of blur kernel" },
		{ GIMP_PDB_INT32, "degree_x", "The degree of polynome (X)" },
		{ GIMP_PDB_INT32, "degree_y", "The degree of polynome (Y)" },
		{ GIMP_PDB_INT32, "nsamples", "The number of samples" },
		{ GIMP_PDB_INT32, "copylf", "Copy low frequencies from template" }
	};
	GimpParamDef *args_out = NULL;
	gint nargs_in = sizeof(args_in) / sizeof(args_in[0]);
	gint nargs_out = 0;
	gimp_install_procedure (PLUGIN_NAME,
				"Texture workshop",
				"Useful operations to convert photos into textures",
				"Lauris Kaplinski",
				"Lauris Kaplinski",
				"2007-2011",
				"<Image>/Filters/Map/Texturewerke...",
				"RGB*, GRAY*",
				GIMP_PLUGIN,
				nargs_in, nargs_out,
				args_in, args_out);
}

static int process_channel (guchar *dpx, int drs, int dbpp, const guchar *spx, int srs, int sbpp, int width, int height, const guchar *mpx, int mrs, int mbpp, const HParams *params);

static void
run (const gchar *name, gint nparams_in, const GimpParam *params_in, gint *nparams_out, GimpParam **params_out)
{
	static GimpParam values[1];

	// Unless anything goes wrong, result is success
	*nparams_out = 1;
	*params_out = values;
	values[0].type = GIMP_PDB_STATUS;
	values[0].data.d_status = GIMP_PDB_SUCCESS;

	HParams hparams = {
		METHOD_LOWPASS_POLYNOME,
		// Drawables
		-1, -1, -1, -1,
		// Color
		false, 2, 2,
		// Blur
		16, 16, // Blur size
		// Lowpas parameters
		64, 64, // Kernel size
		2, // Kernel power
		3, 3, // Degree
		10000, // Number of samples
		false // Copy low frequencies
	};

	int runmode = params_in[PARAM_RUNMODE].data.d_int32;
	int layer_id = params_in[PARAM_DRAWABLE].data.d_drawable;
	if (!gimp_drawable_is_valid (layer_id) || !gimp_drawable_is_layer (layer_id)) {
		if (runmode == GIMP_RUN_INTERACTIVE) {
			gimp_ui_init ("TextureWerke", 0);
			GtkWidget *dialog = gtk_message_dialog_new (NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, "Wrong or invalid drawable selected. TextureWerke works only with layer images.");
			gtk_dialog_run (GTK_DIALOG(dialog));
		}
		values[0].data.d_status = GIMP_PDB_EXECUTION_ERROR;
		return;
	}
	int mask_id = gimp_layer_get_mask (layer_id);
	// g_log ("TextureWerke", G_LOG_LEVEL_DEBUG, "Drawable %d Mask %d", layer_id, mask_id);

	if (runmode == GIMP_RUN_INTERACTIVE) {
		gimp_get_data (PLUGIN_NAME, &hparams);
		hparams.layer_id = layer_id;
		hparams.mask_id = mask_id;
		unsigned int result = dialog_main_run (&hparams);
		if (!result) return;
	} else if (runmode == GIMP_RUN_NONINTERACTIVE) {
		hparams.method = params_in[PARAM_METHOD].data.d_int32;

		hparams.layer_id = layer_id;
		hparams.mask_id = mask_id;
		int other_layer_id = params_in[PARAM_OTHER_LAYER].data.d_drawable;
		int other_mask_id = -1;
		if (gimp_drawable_is_valid (other_layer_id) || gimp_drawable_is_layer (other_layer_id)) {
			other_mask_id = gimp_layer_get_mask (other_layer_id);
		} else {
			other_layer_id = -1;
		}
		hparams.other_layer_id = other_layer_id;
		hparams.other_mask_id = other_mask_id;

		hparams.color_use_blur = params_in[PARAM_COLOR_USE_BLUR].data.d_int32;
		hparams.polynome_blur_x = hparams.blur_x = hparams.color_blur_x = (float) params_in[PARAM_XSIGMA].data.d_float;
		hparams.polynome_blur_y = hparams.blur_y = hparams.color_blur_y = (float) params_in[PARAM_YSIGMA].data.d_float;
		hparams.polynome_degree_x = params_in[PARAM_DEGREE_X].data.d_int32;
		hparams.polynome_degree_y = params_in[PARAM_DEGREE_Y].data.d_int32;
		hparams.kpower = params_in[PARAM_KPOWER].data.d_float;
		hparams.nsamples = params_in[PARAM_NSAMPLES].data.d_int32;
		hparams.copylf = params_in[PARAM_COPYLF].data.d_int32;
	} else if (runmode == GIMP_RUN_WITH_LAST_VALS) {
		gimp_get_data (PLUGIN_NAME, &hparams);
		hparams.layer_id = layer_id;
		hparams.mask_id = mask_id;
	}

	GimpDrawable *drawable = gimp_drawable_get (hparams.layer_id);
	GimpDrawable *mask = (hparams.mask_id >= 0) ? gimp_drawable_get (hparams.mask_id) : NULL;

	if (hparams.method == METHOD_COLOR) {
		color_process (drawable, mask, &hparams);
	} else {
		polynome_process (drawable, mask, &hparams);
	}

	if (mask) gimp_drawable_detach (mask);
	gimp_drawable_detach (drawable);
#if 0
	int width = drawable->width;
	int height = drawable->height;
	int bpp = drawable->bpp;
	// Source rectangle
	unsigned char *spx = (unsigned char *) g_malloc (width * height * bpp);
	GimpPixelRgn regsrc;
	gimp_pixel_rgn_init (&regsrc, drawable, 0, 0, width, height, 0, 0);
	gimp_pixel_rgn_get_rect (&regsrc, spx, 0, 0, width, height);

	// Get mask
	guchar *mpx = NULL;
	if (mask && (mask->width == width) && (mask->height == height)) {
		mpx = (guchar *) g_malloc(width * height * mask->bpp);
		GimpPixelRgn regmask;
		gimp_pixel_rgn_init (&regmask, mask, 0, 0, width, height, 0, 0);
		gimp_pixel_rgn_get_rect (&regmask, mpx, 0, 0, width, height);
	}

	guchar *dpx = (guchar *) g_malloc (width * height * bpp);

	for (int channel = 0; channel < bpp; channel++) {
		int maskrs = (mask) ? width * mask->bpp : 0;
		int maskbpp = (mask) ? mask->bpp : 0;
		if (!process_channel (dpx + channel, width * bpp, bpp, spx + channel, width * bpp, bpp, width, height, mpx, maskrs, maskbpp, &hparams)) {
			if (mask) gimp_drawable_detach (mask);
			gimp_drawable_detach (drawable);
			values[0].data.d_status = GIMP_PDB_EXECUTION_ERROR;
			if (mpx) g_free (mpx);
			g_free (dpx);
			g_free (spx);
			return;
		}
		gimp_progress_update (1.0 * channel / bpp);
	}

	// Write dpx to drawable
	GimpPixelRgn regdst;
	gimp_pixel_rgn_init (&regdst, drawable, 0, 0, width, height, 1, 1);
	gimp_pixel_rgn_set_rect (&regdst, dpx, 0, 0, width, height);
      
	gimp_drawable_merge_shadow (drawable->drawable_id, 1);
	gimp_drawable_update (drawable->drawable_id, 0, 0, width, height);
#endif

	if (runmode != GIMP_RUN_NONINTERACTIVE) {
		gimp_displays_flush ();
	}
   
	if (runmode == GIMP_RUN_INTERACTIVE) {
		gimp_set_data (PLUGIN_NAME, &hparams, sizeof (hparams));
	}

#if 0
	// Detach drawables
	if (mask) gimp_drawable_detach (mask);
	gimp_drawable_detach (drawable);

	if (mpx) g_free (mpx);
	g_free (dpx);
	g_free (spx);
#endif
}

#if 0
static int
process_channel (guchar *dpx, int drs, int dbpp, const guchar *spx, int srs, int sbpp, int width, int height, const guchar *mpx, int mrs, int mbpp, const HParams *params)
{
	if (params->type == TYPE_SYMMETRIC_FT) {
		process_ft (dpx, drs, dbpp, spx, srs, sbpp, width, height, mpx, mrs, mbpp, (float) params->hfreqsigma, (float) params->vfreqsigma);
	} else if (params->type == TYPE_ASYMMETRIC_FT) {
	} else {
		process_channel_lm (dpx, drs, dbpp, spx, srs, sbpp, width, height, NULL, 0, 0, params);
	}
	return 1;
}
#endif

