// Rainbow tonemapping
// gimp plugin
// to install (on linux):
// for user:
// gimptool-2.0 --install tonemapping.c
//and for system-wide installation:
// gimptool-2.0 --install-admin tonemapping.c
//contact: tiborb95 at gmail dot com, any feedback welcomed


#include <libgimp/gimp.h>
#include <stdio.h>
#include <libgimp/gimpui.h>
#include <string.h>
#include <stdlib.h> // for exit
#include <math.h>


#define VERSION 0.2
#define VERBOSE FALSE
#define DEBUG FALSE
#define DEBUGVERBOSE FALSE
//#define DEBUGKILL TRUE
#define RESPONSE_RESET   99
#define RESPONSE_OK      1
#define TO_FLOAT(x) pow((gfloat)x/255,1/2.2)
#define GAMME_TO_INT(x) MAX(MIN(ROUND(pow(x,2.2)*255),255),0)
#define BACK_TO_INT(x) MAX(MIN(x*255,255),0)
#define SQ(x) pow(x,2)
#define ITERATE(x,start,end) for(x=start;x<end;x++)


//structures
typedef struct {
	gfloat Rs; // data are in range 0-255, but can be floats
	gfloat Gs;
	gfloat Bs;
	gfloat Re;
	gfloat Ge;
	gfloat Be;
	gfloat lenght;
	gint mode;
	gboolean preview;
} MyGenVals;
static MyGenVals maindata = {0,0,0,255,255,255,80,1};
GimpRGB startRGB = {0,0,0};	//defined in floats 0-1
GimpRGB endRGB = {1,1,1};	
//variables
gboolean previewbool;
static gint channels;
static gint height, width;
static guchar *rect_in,*rect_out;
gboolean preview;
gboolean process_image = FALSE;
static gfloat	gammed_values[256];

//GUI variables
gchar *title = "Rainbow tonemapping (0.2)";
static gchar *mode1 = "1 band-average";
static gchar *mode2 = "3 bands - RGB";
GtkWidget *start_color,*end_color;
GtkWidget *combo;
GtkWidget *Rs_spin,*Gs_spin,*Bs_spin;
GtkWidget *Re_spin,*Ge_spin,*Be_spin;
GtkWidget *lenght_spin;


//functions
static void query (void);
static void run   (const gchar      *name,
                   gint              nparams,
                   const GimpParam  *param,
                   gint             *nreturn_vals,
                   GimpParam       **return_vals);
static void tonemap  (GimpDrawable *drawable,GimpPreview *preview);
GimpPlugInInfo PLUG_IN_INFO = { NULL, NULL, query, run };
static gboolean tm_dialog (GimpDrawable *drawable);
static gfloat get_bright (gfloat R,gfloat G,gfloat B);
static void response_callback (GtkWidget *widget,  gint response_id);
static void  mode_changed( GtkComboBox *combo, gpointer data );
static void generate_random(void);
inline gint basepos(gint x, gint y,gint ch, gint width);
inline gfloat get_abs (gfloat x);
static void rgbupdate(GimpPreview *preview);




MAIN()


static void query (void) {
	static GimpParamDef args[] =   {
  		{ GIMP_PDB_INT32, "run-mode", "Run mode" },
    	{ GIMP_PDB_IMAGE, "image", "Input image"  },
    	{ GIMP_PDB_DRAWABLE,"drawable", "Input drawable" }
    };

	gimp_install_procedure (
    	"plug-in-tonemap",
    	title,
    	"Rainbow tonemapping - remap colors of image",
    	"Tibor Bamhor",
    	"GPL v.3",
    	"2011",
    	"_Rainbow Tonemap",
    	"RGB*",
    	GIMP_PLUGIN,
    	G_N_ELEMENTS (args), 0,
    	args, NULL);

	gimp_plugin_menu_register ("plug-in-tonemap",                        
    	"<Image>/Filters/Artistic");
}


static void run (const gchar      *name,
     			gint              nparams,
     			const GimpParam  *param,
     			gint             *nreturn_vals,
     			GimpParam       **return_vals){
	static GimpParam  values[1];
	GimpPDBStatusType status = GIMP_PDB_SUCCESS;
	GimpRunMode       run_mode;
	GimpDrawable     *drawable;

	/* Setting mandatory output values */
	*nreturn_vals = 1;
	*return_vals  = values;

	values[0].type = GIMP_PDB_STATUS;
	values[0].data.d_status = status;

	// Getting run_mode - we won't display a dialog if
	// we are in NONINTERACTIVE mode
  	run_mode = param[0].data.d_int32;

  	/*  Get the specified drawable  */
	drawable = gimp_drawable_get (param[2].data.d_drawable);




    switch (run_mode) {
    	case GIMP_RUN_INTERACTIVE:
            /* Get options last values if needed */
            //gimp_get_data ("plug-in-tonemap", &satvals);

            /* Display the dialog */
            if (! tm_dialog (drawable)) return;
            break;

		case GIMP_RUN_NONINTERACTIVE:
            if (nparams != 4)  status = GIMP_PDB_CALLING_ERROR;
            //if (status == GIMP_PDB_SUCCESS) satvals.value1 = param[3].data.d_int32;
            //???????????????
            break;

		//case GIMP_RUN_WITH_LAST_VALS:
            ///*  Get options last values if needed  */
            //gimp_get_data ("plug-in-tonemap", &satvals);
            //break;

		default:
            break;
        }

	//gimp_progress_init ("Saturating...");

	tonemap (drawable,NULL);


	gimp_displays_flush ();
	gimp_drawable_detach (drawable);

	return;
}

inline gint basepos(gint x, gint y,gint ch, gint width){
	return ch*y*width + ch*x;}
	
inline gfloat get_abs (gfloat x) {
	if (x<0) x=-x;
	return x;}

static gfloat get_bright (gfloat R,gfloat G,gfloat B){
	gfloat avg;
	
	avg= 0.299*R + 0.587*G + 0.114*B;
	
    return avg;}

void rgbupdate(GimpPreview *preview) {
	//updates RGB structs and color selectors
	
	gimp_rgb_set_uchar (&startRGB,maindata.Rs,maindata.Gs,maindata.Bs);
	gimp_rgb_set_uchar (&endRGB,maindata.Re,maindata.Ge,maindata.Be);
	gimp_color_area_set_color (GIMP_COLOR_AREA (start_color),&startRGB);
	gimp_color_area_set_color (GIMP_COLOR_AREA (end_color),&endRGB);	
	gimp_preview_invalidate(preview);
	
}


void  mode_changed( GtkComboBox *combo, gpointer data ) {
    

    if ( strncmp( gtk_combo_box_get_active_text( combo ), mode1,10) == 0) {
    	maindata.mode=0;}
    if ( strncmp((gtk_combo_box_get_active_text( combo )),mode2,10) == 0) {
    	maindata.mode=1;}

    if (VERBOSE) printf( "mode_changed: Changing type of mask to %d\n",maindata.mode );
}

void generate_random() {

     gtk_spin_button_set_value(GTK_SPIN_BUTTON(Rs_spin),rand()% 140);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(Gs_spin),rand()% 140);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(Bs_spin),rand()% 140);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(Re_spin),rand()% 145 +110);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(Ge_spin),rand()%  145 +110);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(Be_spin),rand()%  145 +110);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(lenght_spin),rand()% 60 + 15);	
 }

	
static void fill_rect_out(gint width, gint height,GimpPreview *preview) {

	gfloat Ro,Go,Bo; // original relative values from pixel  (gammed to be usable for processing)
	gfloat Rn,Gn,Bn;  //new (ungammed) values
	gint x,y;
	gint Rint,Gint,Bint;
	gfloat avg;
	gint basepos_out;
	gfloat modulo;
	gfloat weight1, weight2, weight3; // weight of start color, (1-weight) is weight of end color
	gfloat Rs,Gs,Bs,Re,Ge,Be,period; //taken from maindata but recaluclated to range 0-1
	
	//gfloat edist; //euklidian distance of RGB
	Rs=TO_FLOAT(maindata.Rs);
	Gs=TO_FLOAT(maindata.Gs);
	Bs=TO_FLOAT(maindata.Bs);
	Re=TO_FLOAT(maindata.Re);
	Ge=TO_FLOAT(maindata.Ge);
	Be=TO_FLOAT(maindata.Be);	
	period=(gfloat) maindata.lenght/255;	
	
		ITERATE(y,0,height) { ITERATE(x,0,width) {
			
						
			//basepos_tmp=basepos(x,y,3,width);
			basepos_out=basepos(x,y,channels,width);
			
			//this reads original data
			Ro=gammed_values[rect_in[basepos_out]];
			Go=gammed_values[rect_in[basepos_out+1]];
			Bo=gammed_values[rect_in[basepos_out+2]];
			if (DEBUGVERBOSE) printf ("old RGB: %.3f. %.3f, %.3f\n",Ro,Go,Bo);
			
			//now algorithme branches
			if (maindata.mode == 0) {
				//1D processing
				avg=get_bright(Ro,Go,Bo);
				modulo=fmod (avg,period) ;

				weight1 = (gfloat) MIN(modulo,period-modulo)*2/period;
				weight2=weight1;
				weight3=weight1;

				if (DEBUGVERBOSE) printf ("modulo : %.3f, weight1: %.3f\n",modulo,weight1);
				}
			else if (maindata.mode == 1) {
				modulo=fmod (Ro,period) ;
				weight1 = (gfloat) MIN(modulo,period-modulo)*2/period;
				modulo=fmod (Go,period) ;
				weight2 = (gfloat) MIN(modulo,period-modulo)*2/period;
				modulo=fmod (Bo,period) ;
				weight3 = (gfloat) MIN(modulo,period-modulo)*2/period;				
				}
			else {
				printf ("Unknown mode, quitting\n");
				exit (1);
				}		
				
			Rn=(gfloat) weight1*Rs+(1-weight1)*Re;
			Gn=(gfloat) weight2*Gs+(1-weight2)*Ge;
			Bn=(gfloat) weight3*Bs+(1-weight3)*Be;
			if (DEBUGVERBOSE) printf ("new RGB: %.3f. %.3f, %.3f\n",Rn,Gn,Bn);			
			//now it should calculate new values

			
			Rint=BACK_TO_INT(Rn);
			Gint=BACK_TO_INT(Gn);
			Bint=BACK_TO_INT(Bn);

			if (DEBUGVERBOSE) printf ("new RGB integers: %.d, %.d, %.d\n",Rint,Gint,Bint);

			rect_out[basepos_out]=Rint;
			rect_out[basepos_out+1]=Gint;
			rect_out[basepos_out+2]=Bint;

			if (DEBUG) {
				if ( rect_out[basepos_out] < MIN(maindata.Rs,maindata.Re) ) printf ("Error R min -out of range\n");
				if ( rect_out[basepos_out+1] < MIN(maindata.Gs,maindata.Ge) ) printf ("Error G min-out of range\n");
				if ( rect_out[basepos_out+2] < MIN(maindata.Bs,maindata.Be) ) printf ("Error B min-out of range\n");
				if ( rect_out[basepos_out]   > MAX(maindata.Rs,maindata.Re) ) printf ("Error R max-out of range\n");
				if ( rect_out[basepos_out+1] > MAX(maindata.Gs,maindata.Ge) ) printf ("Error G max-out of range\n");
				if ( rect_out[basepos_out+2] > MAX(maindata.Bs,maindata.Be) ) printf ("Error B max-out of range\n");}				

			if (channels==4) rect_out[basepos_out+3] = rect_in[basepos_out+3]; 
			if (DEBUGVERBOSE && channels==4) printf ("Copying alpha channel: %.d\n",rect_out[basepos_out+4]);
		}
		if (!preview && y % 10 ==0 ) gimp_progress_update( (gdouble) y / height);
		}
		
	}

static void tonemap (GimpDrawable *drawable,GimpPreview *preview) {

	gint         x1, y1, x2, y2; 		//x,y from image not area
	GimpPixelRgn rgn_in, rgn_out;
 	//gint         width, height;
	gint i;

//
	if (VERBOSE) printf ("Running tonemap()\n");
	if (!preview) gimp_progress_init ("Rainbow tonemapping...");

	// getting coordinates of what will be saturated
	if (preview){
          gimp_preview_get_position (preview, &x1, &y1);
          gimp_preview_get_size (preview, &width, &height);
          x2 = x1 + width;
          y2 = y1 + height;

			}
    else {
         gimp_drawable_mask_bounds (drawable->drawable_id,&x1, &y1,&x2, &y2);
         width = x2 - x1;
         height = y2 - y1;}
         
   
 channels = gimp_drawable_bpp (drawable->drawable_id);//getting number of channels

	for (i=0;i<256;i++) { //to speed up conversion from raw RGB to gammed values
		gammed_values[i]=pow((gfloat)i/255,1/2.2);}


  //initializing pixelrgn for input and output row
  gimp_pixel_rgn_init (&rgn_in,
                       drawable,
                       x1, y1,
                       width, height,
                       FALSE, FALSE); 
  gimp_pixel_rgn_init (&rgn_out,
                       drawable,
                       x1, y1,
                       width, height,
                       preview == NULL, TRUE);

	if (VERBOSE) printf ("Processed area size: %.d x %.d, channels: %.d\n", width,height,channels);
	
 	/* Initialise memory for row1 */
	rect_in = g_new (guchar, channels  * width * height); 
	rect_out = g_new (guchar, channels * width * height);

	//loading all data to rect_in
	gimp_pixel_rgn_get_rect (&rgn_in,rect_in,x1, y1,width,height);

	//finding current RGB values in samples
	//gimp_rgb_get_uchar (&endRGB,&R2,&G2,&B2);

	//calculating data
	fill_rect_out(width,height, GIMP_PREVIEW (preview));

	//putting data back to drawable
    gimp_pixel_rgn_set_rect (&rgn_out, rect_out,x1, y1,width,height);

  g_free (rect_in);
  g_free (rect_out);

  //update (graphical) of preview/image
	if (preview) {
    	gimp_drawable_preview_draw_region (GIMP_DRAWABLE_PREVIEW (preview),
        &rgn_out);}
    else {
		gimp_drawable_flush (drawable);
		gimp_drawable_merge_shadow (drawable->drawable_id, TRUE);
		gimp_drawable_update (drawable->drawable_id,x1,y1,width,height);}
}



static gboolean
tm_dialog (GimpDrawable *drawable)
{
  GtkWidget *dialog;
  GtkWidget *main_vbox,*table1, *table2;
  GtkWidget *start_label, *end_label, *combo_label;

  GtkWidget *Rs_label,*Gs_label, *Bs_label;
  GtkWidget *Re_label,*Ge_label, *Be_label;

  GtkObject *Rs_spin_adj,*Gs_spin_adj,*Bs_spin_adj;
  GtkObject *Re_spin_adj,*Ge_spin_adj,*Be_spin_adj;
  GtkWidget *preview;
  GtkWidget *lenght_label;

  GtkObject *lenght_spin_adj;
  GtkWidget *randombutton;



  gimp_ui_init ("Rainbow Tonemapperr", FALSE);
  
  gimp_dialogs_show_help_button (FALSE);
  dialog = gimp_dialog_new (title, "tonemap",
                            NULL, 0,
                            gimp_standard_help_func, "plug-in-tonemap",

                            GIMP_STOCK_RESET, RESPONSE_RESET,
                            GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                            GTK_STOCK_OK,     RESPONSE_OK,

                            NULL);

  
  gtk_container_set_border_width (GTK_CONTAINER (dialog), 5); // ???

  
  main_vbox = gtk_vbox_new (FALSE, 10);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), main_vbox);
  gtk_widget_show (main_vbox);
  
  preview = gimp_drawable_preview_new (drawable, &previewbool);
  gtk_box_pack_start (GTK_BOX (main_vbox), preview, TRUE, TRUE, 0);
  gtk_widget_show (preview);

	//creating table 1
	table1=gtk_table_new(4,5,FALSE);
	gtk_table_set_col_spacings( GTK_TABLE(table1), 5 );
	gtk_table_set_row_spacings( GTK_TABLE(table1), 3 );
	gtk_box_pack_start (GTK_BOX (main_vbox), table1, FALSE, FALSE, 5);

		//inserting labels
		start_label = gtk_label_new ("<b>Start color:</b>");
		gtk_label_set_use_markup (GTK_LABEL(start_label), TRUE);
		gtk_misc_set_alignment (GTK_MISC (start_label), 0.1, 0);
		gtk_table_attach_defaults(GTK_TABLE(table1),start_label,0,2,0,1);
		gtk_widget_show (start_label);
		end_label = gtk_label_new ("<b>End color:</b>");
		gtk_label_set_use_markup (GTK_LABEL(end_label), TRUE);
		gtk_misc_set_alignment (GTK_MISC (end_label), 0.1, 0);
		gtk_table_attach_defaults(GTK_TABLE(table1),end_label,2,4,0,1);
		gtk_widget_show (end_label);

  		//inserting color areas
  		start_color=gimp_color_area_new(&startRGB,GIMP_COLOR_AREA_FLAT, GDK_BUTTON1_MASK | GDK_BUTTON2_MASK);
  		gtk_widget_set_size_request(start_color,30,30);
  		gimp_color_area_set_draw_border(GIMP_COLOR_AREA(start_color),TRUE);
		gtk_table_attach_defaults(GTK_TABLE(table1),start_color,0,2,1,2);
  		gtk_widget_show (start_color);
  		end_color=gimp_color_area_new(&endRGB,GIMP_COLOR_AREA_FLAT, GDK_BUTTON1_MASK | GDK_BUTTON2_MASK);
		gtk_table_attach_defaults(GTK_TABLE(table1),end_color,2,4,1,2);
		 gimp_color_area_set_draw_border(GIMP_COLOR_AREA(end_color),TRUE);
  		gtk_widget_show (end_color);
  		
  		// creating RGB labels (x2)
  		Rs_label = gtk_label_new ("R");
 		gtk_table_attach_defaults(GTK_TABLE(table1),Rs_label,0,1,2,3);
   		gtk_widget_show (Rs_label);		
  		Gs_label = gtk_label_new ("G");
 		gtk_table_attach_defaults(GTK_TABLE(table1),Gs_label,0,1,3,4);
   		gtk_widget_show (Gs_label);	
  		Bs_label = gtk_label_new ("B");
 		gtk_table_attach_defaults(GTK_TABLE(table1),Bs_label,0,1,4,5);
   		gtk_widget_show (Bs_label);			
  		Re_label = gtk_label_new ("R");
 		gtk_table_attach_defaults(GTK_TABLE(table1),Re_label,2,3,2,3);
   		gtk_widget_show (Re_label);		
  		Ge_label = gtk_label_new ("G");
 		gtk_table_attach_defaults(GTK_TABLE(table1),Ge_label,2,3,3,4);
   		gtk_widget_show (Ge_label);	
  		Be_label = gtk_label_new ("B");
 		gtk_table_attach_defaults(GTK_TABLE(table1),Be_label,2,3,4,5);
   		gtk_widget_show (Be_label);	
   		
   		//spinboxes for RGB (x2)
		Rs_spin_adj = gtk_adjustment_new (maindata.Rs, 0, 255, 1, 10,0);
		Rs_spin = gtk_spin_button_new (GTK_ADJUSTMENT (Rs_spin_adj), 2, 1);
		gtk_table_attach_defaults(GTK_TABLE(table1),Rs_spin,1,2,2,3);
		gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (Rs_spin), TRUE);
		gtk_widget_show (Rs_spin);	
		Gs_spin_adj = gtk_adjustment_new (maindata.Gs,  0, 255, 1, 10,0);
		Gs_spin = gtk_spin_button_new (GTK_ADJUSTMENT (Gs_spin_adj),2, 1);
		gtk_table_attach_defaults(GTK_TABLE(table1),Gs_spin,1,2,3,4);
		gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (Gs_spin), TRUE);
		gtk_widget_show (Gs_spin);	
		Bs_spin_adj = gtk_adjustment_new (maindata.Bs, 0, 255, 1, 10,0);
		Bs_spin = gtk_spin_button_new (GTK_ADJUSTMENT (Bs_spin_adj), 2, 1);
		gtk_table_attach_defaults(GTK_TABLE(table1),Bs_spin,1,2,4,5);
		gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (Bs_spin), TRUE);
		gtk_widget_show (Bs_spin);	
		Re_spin_adj = gtk_adjustment_new (maindata.Re,  0, 255, 1, 10,0);
		Re_spin = gtk_spin_button_new (GTK_ADJUSTMENT (Re_spin_adj),2, 1);
		gtk_table_attach_defaults(GTK_TABLE(table1),Re_spin,3,4,2,3);
		gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (Re_spin), TRUE);
		gtk_widget_show (Re_spin);	
		Ge_spin_adj = gtk_adjustment_new (maindata.Ge,  0, 255, 1, 10,0);
		Ge_spin = gtk_spin_button_new (GTK_ADJUSTMENT (Ge_spin_adj),2, 1);
		gtk_table_attach_defaults(GTK_TABLE(table1),Ge_spin,3,4,3,4);
		gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (Ge_spin), TRUE);
		gtk_widget_show (Ge_spin);	
		Be_spin_adj = gtk_adjustment_new (maindata.Be,  0, 255, 1, 10,0);
		Be_spin = gtk_spin_button_new (GTK_ADJUSTMENT (Be_spin_adj), 2, 1);
		gtk_table_attach_defaults(GTK_TABLE(table1),Be_spin,3,4,4,5);
		gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (Be_spin), TRUE);
		gtk_widget_show (Be_spin);	

	gtk_widget_show (table1);	

	//creating table 2
	table2=gtk_table_new(2,2,FALSE);
	gtk_box_pack_start (GTK_BOX (main_vbox), table2, FALSE, FALSE, 0);
	
		//inserting lenght label
		lenght_label = gtk_label_new ("<b>Period lenght:</b>");
		gtk_label_set_use_markup (GTK_LABEL(lenght_label), TRUE);
		gtk_misc_set_alignment (GTK_MISC (lenght_label), 0.1, 0.5);
		gtk_table_attach_defaults(GTK_TABLE(table2),lenght_label,0,1,0,1);
		gtk_widget_show (lenght_label);	
		
		//lenght spin
		lenght_spin_adj = gtk_adjustment_new (maindata.lenght, 0, 255, 0.1, 1,1);
		lenght_spin = gtk_spin_button_new (GTK_ADJUSTMENT (lenght_spin_adj), 2, 1);
		gtk_table_attach_defaults(GTK_TABLE(table2),lenght_spin,1,2,0,1);
		gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (lenght_spin), TRUE);
		gtk_widget_show (lenght_spin);	

		//label for combo
		combo_label= gtk_label_new ("<b>Tonemap mode:</b>");
		gtk_label_set_use_markup (GTK_LABEL(combo_label), TRUE);		
		gtk_misc_set_alignment (GTK_MISC (combo_label), 0.1, 0.5);
		gtk_table_attach_defaults(GTK_TABLE(table2),combo_label,0,1,1,2);
		gtk_widget_show (combo_label);	
		
		//creating combo for selection of tonemap mode
		combo = gtk_combo_box_new_text();
		gtk_combo_box_append_text( GTK_COMBO_BOX( combo),mode1 );
		gtk_combo_box_append_text( GTK_COMBO_BOX( combo),mode2);
		gtk_table_attach_defaults(GTK_TABLE(table2),combo,1,2,1,2);
		gtk_combo_box_set_active(GTK_COMBO_BOX( combo), maindata.mode);
		gtk_widget_show (combo);		
		
	gtk_widget_show (table2);			
	
	randombutton = gtk_button_new_with_label("Random values");
	gtk_box_pack_start (GTK_BOX (main_vbox), randombutton, FALSE, FALSE, 0);
	gtk_widget_show (randombutton);
	
	
  g_signal_connect_swapped (preview, "invalidated",  G_CALLBACK (tonemap),drawable);

  g_signal_connect_swapped (lenght_spin_adj, "value_changed", G_CALLBACK (gimp_preview_invalidate), preview);
  g_signal_connect_swapped (G_OBJECT( combo ), "changed", G_CALLBACK (gimp_preview_invalidate), preview);
  g_signal_connect_swapped (G_OBJECT (randombutton), "clicked", G_CALLBACK (gimp_preview_invalidate), preview);
 
 tonemap (drawable, GIMP_PREVIEW (preview));

  //update of values in maindata
  g_signal_connect (Rs_spin_adj, "value_changed", G_CALLBACK (gimp_float_adjustment_update), &maindata.Rs);  
  g_signal_connect (Gs_spin_adj, "value_changed", G_CALLBACK (gimp_float_adjustment_update), &maindata.Gs);  
  g_signal_connect (Bs_spin_adj, "value_changed", G_CALLBACK (gimp_float_adjustment_update), &maindata.Bs);  
  g_signal_connect (Re_spin_adj, "value_changed", G_CALLBACK (gimp_float_adjustment_update), &maindata.Re);  
  g_signal_connect (Ge_spin_adj, "value_changed", G_CALLBACK (gimp_float_adjustment_update), &maindata.Ge);  
  g_signal_connect (Be_spin_adj, "value_changed", G_CALLBACK (gimp_float_adjustment_update), &maindata.Be); 
  
  //updating rgbstructs and color samples
  g_signal_connect_swapped (Rs_spin_adj, "value_changed", G_CALLBACK (rgbupdate), preview);  
  g_signal_connect_swapped (Gs_spin_adj, "value_changed", G_CALLBACK (rgbupdate), preview);  
  g_signal_connect_swapped (Bs_spin_adj, "value_changed", G_CALLBACK (rgbupdate), preview);  
  g_signal_connect_swapped (Re_spin_adj, "value_changed", G_CALLBACK (rgbupdate), preview);  
  g_signal_connect_swapped (Ge_spin_adj, "value_changed", G_CALLBACK (rgbupdate), preview);  
  g_signal_connect_swapped (Be_spin_adj, "value_changed", G_CALLBACK (rgbupdate), preview); 
  
  //updating period/lenght
  g_signal_connect (lenght_spin_adj, "value_changed", G_CALLBACK (gimp_float_adjustment_update), &maindata.lenght);  
    
  //updating combo status
   g_signal_connect( G_OBJECT( combo ), "changed",G_CALLBACK( mode_changed ), NULL );

	//generating random values
	  g_signal_connect( G_OBJECT(randombutton),"clicked",G_CALLBACK( generate_random ), NULL );

  gtk_widget_show (dialog);
  
  //receiving signals from gimp buttons + "x" (close window button)
  g_signal_connect (dialog, "response",
                    G_CALLBACK (response_callback),
                    NULL);

  gtk_main ();

  return process_image; //TRUE if image should be processed

}

static void response_callback (GtkWidget *widget,  gint response_id) {

	
  switch (response_id) 	 {
    case RESPONSE_RESET:
      printf ("Resetting... \n");

	gtk_spin_button_set_value(GTK_SPIN_BUTTON(Rs_spin),0);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(Gs_spin),0);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(Bs_spin),0);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(Re_spin),255);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(Ge_spin),255);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(Be_spin),255);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(lenght_spin),80);	


          
      break;

    case RESPONSE_OK:  //quitting plugin window and applying change
      //printf ("OK... \n");
      process_image=TRUE; 
      gtk_widget_destroy (widget);
      gtk_main_quit ();     
      break;

    default: // if other response - terminate plugin window
      gtk_widget_destroy (widget);
      gtk_main_quit ();
      break;
    };
	
}

