;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                         ;;;
;;; GIMP - The GNU Image Manipulation Program                               ;;;
;;; Copyright (C) 1995 Spencer Kimball and Peter Mattis                     ;;;
;;;                                                                         ;;;
;;; This program is free software: you can redistribute it and/or modify    ;;;
;;; it under the terms of the GNU General Public License as published by    ;;;
;;; the Free Software Foundation, either version 3 of the License, or       ;;;
;;; (at your option) any later version.                                     ;;;
;;;                                                                         ;;;
;;; This program is distributed in the hope that it will be useful,         ;;;
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of          ;;;
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           ;;;
;;; GNU General Public License for more details.                            ;;;
;;;                                                                         ;;;
;;; You should have received a copy of the GNU General Public License       ;;;
;;; along with this program.  If not, see <http://www.gnu.org/licenses/>.   ;;;
;;;                                                                         ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                         ;;;
;;; Translate into Palette                                                  ;;;
;;;                                                                         ;;;
;;; translate-into-palette.scm - version 1.00                               ;;;
;;; Copyright (C) 2014 Gino D                                               ;;;
;;; https://sites.google.com/site/ginodonig/gimp-scripts                    ;;;
;;;                                                                         ;;;
;;; This script creates a new palette whose display area provides a         ;;;
;;; faithful representation of the specified contents of the current image. ;;;
;;;                                                                         ;;;
;;; ....................................................................... ;;;
;;;                                                                         ;;;
;;; VERSION HISTORY                                                         ;;;
;;;                                                                         ;;;
;;; 1.00 - January 2014                                                     ;;;
;;;  * Initial release.                                                     ;;;
;;;                                                                         ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define *translate-into-palette-2.8.0* 1)

(let* ((gimp-version-comparison
        (lambda (predicate version-a version-b)
          (letrec ((func
                    (lambda (string1 p q)
                      (cond ((= (string-length string1) p)
                             (* (expt 100 q) (string->atom string1)))
                            ((eqv? (string-ref string1 p)
                                   (integer->char 46))
                             (+ (* (expt 100 q)
                                   (string->atom (substring string1 0 p)))
                                (func (substring string1
                                                 (+ p 1)
                                                 (string-length string1))
                                      0
                                      (- q 1))))
                            (else (func string1 (+ p 1) q))))))
            (predicate (func version-a 0 2)
                       (func version-b 0 2))))))
  (cond ((gimp-version-comparison < (car (gimp-version)) "2.6.10")
         (quit))
        ((gimp-version-comparison < (car (gimp-version)) "2.8.0")
         (set! *translate-into-palette-2.8.0* 0))))

(if (not (defined? (quote *tip-plt-fnm*)))
    (define *tip-plt-fnm* ""))

(if (not (defined? (quote *tip-crt-img*)))
    (define *tip-crt-img* 0))

(define (script-fu-translate-into-palette img1
                                          img1-drw1
                                          source
                                          make-opaque
                                          interp-method
                                          palette-name
                                          num-cols
                                          limit-colors
                                          output-folder
                                          del-last-palette)
  (cond ((= *translate-into-palette-2.8.0* 0)
         (define (gimp-image-insert-layer image layer parent position)
           (gimp-image-add-layer image layer position))))
  (define (round2 num int)
    (let* ((rnd (/ (truncate (+ (* (expt 10 int) num)
                                (if (< num 0) -0.5 0.5)))
                   (expt 10 int))))
      (if (= int 0)
          (inexact->exact rnd)
          rnd)))
  (define (color-notations color)
    (let* ((func (lambda (x)
                   (string-append
                    (make-string 1
                                 (string-ref "0123456789ABCDEF"
                                             (quotient x 16)))
                    (make-string 1
                                 (string-ref "0123456789ABCDEF"
                                             (remainder x 16)))))))
      (string-append "#"
                     (func (car color))
                     (func (car (cdr color)))
                     (func (car (cdr (cdr color))))
                     " - "
                     "RGB("
                     (atom->string (car color))
                     ","
                     (atom->string (car (cdr color)))
                     ","
                     (atom->string (car (cdr (cdr color))))
                     ")")))
  (let* ((img1-fname (car (gimp-image-get-filename img1)))
         (img1-sln (car (gimp-image-get-selection img1)))
         (img1-sln-bounds
          (apply vector
                 (case source
                   ((0) (gimp-selection-bounds img1))
                   ((1) (gimp-drawable-mask-bounds img1-drw1)))))
         (img1-sln-x1 (vector-ref img1-sln-bounds 1))
         (img1-sln-y1 (vector-ref img1-sln-bounds 2))
         (img1-sln-x2 (vector-ref img1-sln-bounds 3))
         (img1-sln-y2 (vector-ref img1-sln-bounds 4))
         (width (- img1-sln-x2 img1-sln-x1))
         (height (- img1-sln-y2 img1-sln-y1))
         (img2 (car (gimp-image-new width height RGB)))
         (img2-drw1
          (case source
            ((0) (car (gimp-layer-new-from-visible img1 img2 "")))
            ((1) (car (gimp-layer-new-from-drawable img1-drw1 img2)))))
         (ratio (/ height width))
         (num-cols (round2 num-cols 0))
         (num-rows (round2 (* num-cols ratio) 0))
         (slash (string-ref gimp-locale-directory
                            (- (string-length gimp-locale-directory)
                               7)))
         (palette-directory
          (string-append (case output-folder
                           ((0) gimp-directory)
                           ((1) gimp-data-directory))
                         (make-string 1 slash)
                         "palettes"
                         (make-string 1 slash)))
         (suffix)
         (plt-oport))
    (cond ((and (= limit-colors TRUE)
                (> (* num-cols num-rows) 256))
           (set! num-cols (round2 (/ 16 (sqrt ratio)) 0))
           (set! num-rows (round2 (* num-cols ratio) 0))
           (cond ((> (* num-cols num-rows) 256)
                  (set! num-cols (- num-cols 1))
                  (set! num-rows (round2 (* num-cols ratio) 0))))))
    (gimp-context-push)
    (gimp-image-undo-disable img2)
    (gimp-image-insert-layer img2 img2-drw1 0 0)
    (gimp-layer-add-alpha img2-drw1)
    (gimp-edit-blend img2-drw1
                     FG-BG-RGB-MODE
                     (case make-opaque
                       ((0 1) BEHIND-MODE)
                       ((2)   25))
                     GRADIENT-RADIAL
                     100
                     0
                     REPEAT-NONE
                     (case make-opaque
                       ((0)   TRUE)
                       ((1 2) FALSE))
                     FALSE
                     1
                     0
                     FALSE
                     0
                     0
                     0
                     0)
    (gimp-layer-translate img2-drw1 (- img1-sln-x1) (- img1-sln-y1))
    (gimp-layer-resize img2-drw1
                       width
                       height
                       (- img1-sln-x1)
                       (- img1-sln-y1))
    (gimp-image-scale-full img2 num-cols num-rows interp-method)
    (cond ((= del-last-palette 2))
          ((not (string? *tip-plt-fnm*)))
          ((not (file-exists? *tip-plt-fnm*)))
          ((= del-last-palette 1)
           (file-delete *tip-plt-fnm*))
          ((not (number? *tip-crt-img*)))
          ((= img1 *tip-crt-img*)
           (file-delete *tip-plt-fnm*)))
    (set! *tip-crt-img* img1)
    (set! palette-name
          (cond ((> (string-length palette-name) 0)
                 palette-name)
                ((> (string-length img1-fname) 0)
                 (letrec ((a-func
                           (lambda (a1 a2 a3)
                             (cond ((= a2 0) 0)
                                   ((eqv? (string-ref a1 a2) a3) a2)
                                   (else (a-func a1 (- a2 1) a3))))))
                   (let* ((b1 (string-length img1-fname))
                          (b2 (a-func img1-fname (- b1 1) slash))
                          (b3 (a-func img1-fname (- b1 1) #\.)))
                     (substring img1-fname
                                (+ b2 1)
                                (if (> b3 (+ b2 1)) b3 b1)))))
                (else
                 (car (gimp-image-get-name img1)))))
    (set! suffix
          (let c-loop ((c 0)
                       (c-str-a (string-append palette-directory palette-name))
                       (c-str-b ""))
            (if (file-exists? (string-append c-str-a c-str-b ".gpl"))
                (c-loop (+ c 1)
                        c-str-a
                        (string-append " (" (atom->string (+ c 1)) ")"))
                c-str-b)))
    (set! *tip-plt-fnm*
          (string-append palette-directory palette-name suffix ".gpl"))
    (set! plt-oport (open-output-file *tip-plt-fnm*))
    (set-output-port plt-oport)
    (display (string-append "GIMP Palette\r\nName: "
                            palette-name
                            suffix
                            "\r\nColumns: "
                            (atom->string num-cols)
                            "\r\n#\r\n"
                            "# Original number of columns: "
                            (atom->string num-cols)
                            "\r\n"
                            "#\r\n")
             plt-oport)
    (gimp-progress-init " " 0)
    (gimp-progress-set-text "0 %")
    (let d-loop ((d 0)
                 (d-pixel ())
                 (d-color ())
                 (d-red ())
                 (d-grn ())
                 (d-blu ())
                 (d-percent ()))
      (set! d-pixel
            (car (cdr (gimp-drawable-get-pixel img2-drw1
                                               (modulo d num-cols)
                                               (quotient d num-cols)))))
      (set! d-color
            (list* (vector-ref d-pixel 0)
                   (vector-ref d-pixel 1)
                   (vector-ref d-pixel 2)
                   ()))
      (set! d-red (atom->string (vector-ref d-pixel 0)))
      (set! d-grn (atom->string (vector-ref d-pixel 1)))
      (set! d-blu (atom->string (vector-ref d-pixel 2)))
      (display
       (string-append (make-string (- 3 (string-length d-red)) #\space)
                      d-red
                      " "
                      (make-string (- 3 (string-length d-grn)) #\space)
                      d-grn
                      " "
                      (make-string (- 3 (string-length d-blu)) #\space)
                      d-blu
                      "\t"
                      (color-notations d-color)
                      "\r\n")
       plt-oport)
      (set! d-percent (/ d (* num-cols num-rows)))
      (gimp-progress-set-text (string-append (atom->string
                                              (round2 (* 100 d-percent) 0))
                                             " %"))
      (gimp-progress-update d-percent)
      (if (< d (- (* num-cols num-rows) 1))
          (d-loop (+ d 1) d-pixel d-color d-red d-grn d-blu d-percent)))
    (close-output-port plt-oport)
    (gimp-palettes-refresh)
    (gimp-image-undo-enable img2)
    (gimp-image-delete img2)
    (gimp-context-pop)
    (gimp-context-set-palette (string-append palette-name suffix))))

(let* ((widget-a '("Visible area of active image"
                   "Active layer or channel"))
       (widget-b '("Using background color"
                   "Using foreground color"
                   "Discarding transparency"))
       (widget-c '("Personal 'palettes' folder"
                   "System 'palettes' folder"))
       (widget-d '("If derived from current image"
                   "In any case"
                   "No")))
  ;;
  (script-fu-register
   "script-fu-translate-into-palette"
   _"Translate into Palette..."
   _"Reproduce the selected region on a new palette with the same appearance"
   "Gino D"
   "Gino D"
   "2014"
   "*"
   SF-IMAGE       "Image"                       0
   SF-DRAWABLE    "Drawable"                    0
   SF-OPTION     _"Source to capture"           widget-a
   SF-OPTION     _"Reproduce non-opaque areas"  widget-b
   SF-ENUM       _"Interpolation method"        '("InterpolationType" "cubic")
   SF-STRING     _"Palette name"                ""
   SF-ADJUSTMENT _"Number of columns"           '(16 1 64 1 1 0 0)
   SF-TOGGLE     _"Do not exceed 256 colors"    FALSE
   SF-OPTION     _"Save palette in"             widget-c
   SF-OPTION     _"Delete last created palette" widget-d))

(script-fu-menu-register "script-fu-translate-into-palette"
                         "<Image>/Image")