/*
 *      voronoi-path.c - Voronoi Path plugin for the GIMP
 *      
 *      Copyright 2009 Marko Peric
 *      
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 2 of the License, or
 *      (at your option) any later version.
 *      
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *      
 *      You should have received a copy of the GNU General Public License
 *      along with this program; if not, write to the Free Software
 *      Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 *      MA 02110-1301, USA.
 */
#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>
#include <stdlib.h>

#define PLUG_IN_PROC "plug-in-voronoi-path"
#define PLUG_IN_BINARY "voronoi-path"
#define DELETED -2
#define le 0
#define re 1
#define MIN_DIST sqrt(8) + 0.1
#define EPS 1.0E-03
#define EXS 1.0E-04

static void query(void);
static void run(const gchar      *name,
                gint              nparams,
                const GimpParam  *param,
                gint             *nreturn_vals,
                GimpParam       **return_vals);

GimpPlugInInfo PLUG_IN_INFO =
{
  NULL,
  NULL,
  query,
  run
};

static gint32 cell_site = FALSE;

MAIN()

/*----------------------------------------------------------------------------- 
 *  query  --  called so that the plugin can inform the GIMP about itself 
 *-----------------------------------------------------------------------------
 */
static void query (void)
{
    static GimpParamDef args[] =
    {
        {GIMP_PDB_INT32,    "run-mode",  "Interactive, non-interactive"},
        {GIMP_PDB_IMAGE,    "image",     "Input image"},
        {GIMP_PDB_VECTORS,  "path",      "Input path"},
        {GIMP_PDB_INT32,    "preserve",  "Preserve original path"},
    };

    gimp_install_procedure(
        PLUG_IN_PROC,
        "Create a Voronoi diagram path",
        "Starting from a set of points, generate a Voronoi diagram",
        "Marko Peric",
        "Marko Peric",
        "November 2009",
        "Voronoi Path...",
        "*",
        GIMP_PLUGIN,
        G_N_ELEMENTS(args), 0,
        args, NULL);

    gimp_plugin_menu_register(PLUG_IN_PROC, "<Vectors>");
}

/*=============================================================================
 * START VORONOI CODE
 *===========================================================================*/
/*
* The author of this software is Steven Fortune.  Copyright (c) 1994 by AT&T
* Bell Laboratories.
* Permission to use, copy, modify, and distribute this software for any
* purpose without fee is hereby granted, provided that this entire notice
* is included in all copies of any software which is or includes a copy
* or modification of this software and in all copies of the supporting
* documentation for such software.
* THIS SOFTWARE IS BEING PROVIDED "AS IS", WITHOUT ANY EXPRESS OR IMPLIED
* WARRANTY.  IN PARTICULAR, NEITHER THE AUTHORS NOR AT&T MAKE ANY
* REPRESENTATION OR WARRANTY OF ANY KIND CONCERNING THE MERCHANTABILITY
* OF THIS SOFTWARE OR ITS FITNESS FOR ANY PARTICULAR PURPOSE.
*/

/* 
* This code was originally written by Stephan Fortune in C code.  I, 
* Shane O'Sullivan, have since modified it, encapsulating it in a C++ class 
* and, fixing memory leaks and adding accessors to the Voronoi Edges.
* Permission to use, copy, modify, and distribute this software for any
* purpose without fee is hereby granted, provided that this entire notice
* is included in all copies of any software which is or includes a copy
* or modification of this software and in all copies of the supporting
* documentation for such software.
* THIS SOFTWARE IS BEING PROVIDED "AS IS", WITHOUT ANY EXPRESS OR IMPLIED
* WARRANTY.  IN PARTICULAR, NEITHER THE AUTHORS NOR AT&T MAKE ANY
* REPRESENTATION OR WARRANTY OF ANY KIND CONCERNING THE MERCHANTABILITY
* OF THIS SOFTWARE OR ITS FITNESS FOR ANY PARTICULAR PURPOSE.
*/
typedef struct fnode
{
    struct fnode *nextfree;
} Freenode;

typedef struct fnodearraylist
{
    Freenode              *memory;
    struct fnodearraylist *next;
} FreeNodeArrayList;

typedef struct
{
    Freenode *head;
    gint      nodesize;
} Freelist;

typedef struct
{
    gdouble x,y;
} Point;

/* Structure used both for sites and for vertices */
typedef struct
{
    Point coord;
    gint  sitenbr;
    gint  refcnt;
} Site;

typedef struct
{
    gdouble  a, b, c;
    Site    *ep[2];
    Site    *reg[2];
    gint     edgenbr;
} Edge;

typedef struct gedge
{
    gdouble       x1, y1, x2, y2;
    struct gedge *next;
} Graphedge;

typedef struct hedge
{
    struct hedge *ELleft;
    struct hedge *ELright;
    Edge         *ELedge;
    gint          ELrefcnt;
    gchar         ELpm;
    Site         *vertex;
    gdouble       ystar;
    struct hedge *PQnext;
} Halfedge;

Freelist   hfl;
Halfedge  *ELleftend;
Halfedge  *ELrightend;
gint       ELhashsize;

Halfedge **ELhash;

gint       triangulate, sorted, plot, debug;
gdouble    xmin, xmax, ymin, ymax, deltax, deltay;

Site      *sites;
gint       nsites;
gint       siteidx;
gint       sqrt_nsites;
gint       nvertices;
Freelist   sfl;
Site      *bottomsite;

gint       nedges;
Freelist   efl;
gint       PQhashsize;
Halfedge  *PQhash;
gint       PQcount;
gint       PQmin;

gint       ntry, totalsearch;
gdouble    pxmin, pxmax, pymin, pymax, cradius;
gint       total_alloc;

gdouble    borderMinX, borderMaxX, borderMinY, borderMaxY;
gdouble    minDistanceBetweenSites;

Graphedge *allEdges;
Graphedge *iteratorEdges;

FreeNodeArrayList *allMemoryList;
FreeNodeArrayList *currentMemoryBlock;

gdouble xValues[6] = {100, 120, 350, 400, 450, 700};
gdouble yValues[6] = {100, 400, 500, 300,  80, 380};

GArray* ax1;
GArray* ay1;
GArray* ax2;
GArray* ay2;

void cleanup()
{
    if (sites != NULL) {
        g_free(sites);
        sites = NULL;
    }

    FreeNodeArrayList *current = NULL; 
    FreeNodeArrayList *prev    = NULL;

    current = prev = allMemoryList;

    while(current->next != NULL) {
        prev = current;
        current = current->next;
        g_free(prev->memory);
        g_free(prev);
        prev = NULL;
    }

    if (current != NULL && current->memory != NULL) {
        g_free(current->memory);
        g_free(current);
    }

    allMemoryList = g_new(FreeNodeArrayList, 1);
    allMemoryList->next = NULL;
    allMemoryList->memory = NULL;
    currentMemoryBlock = allMemoryList;
}

void cleanupEdges()
{
    Graphedge *geCurrent = NULL;
    Graphedge *gePrev = NULL;
    geCurrent = gePrev = allEdges;

    while(geCurrent != NULL && geCurrent->next != NULL) {
        gePrev = geCurrent;
        geCurrent = geCurrent->next;
        g_free(gePrev);
    }

    allEdges = NULL;
}

gchar* myalloc(guint n)
{
    gchar *t = NULL;  
    t = (gchar*) g_malloc(n);
    total_alloc += n;
    return t;
}

void freeinit(Freelist *fl, gint size)
{
    fl->head = (Freenode *) NULL;
    fl->nodesize = size;
}

void geominit()
{   
    gdouble sn;

    freeinit(&efl, sizeof(Edge));
    nvertices = 0;
    nedges = 0;
    sn = (gdouble) nsites + 4;
    sqrt_nsites = (gint) sqrt(sn);
    deltay = ymax - ymin;
    deltax = xmax - xmin;
}

gint scomp(const void *p1, const void *p2)
{
    Point *s1 = (Point*) p1;
    Point *s2 = (Point*) p2;
    if (s1->y < s2->y) return -1;
    if (s1->y > s2->y) return 1;
    if (s1->x < s2->x) return -1;
    if (s1->x > s2->x) return 1;
    return 0;
}

gboolean PQinitialize()
{
    gint i; 
    
    PQcount = 0;
    PQmin = 0;
    PQhashsize = 4 * sqrt_nsites;
    PQhash = (Halfedge *) myalloc(PQhashsize * sizeof *PQhash);

    if (PQhash == NULL) 
        return FALSE;

    for (i = 0; i < PQhashsize; i++) 
        PQhash[i].PQnext = (Halfedge *) NULL;

    return TRUE;
}

Site* nextone()
{
    Site *s;
    if(siteidx < nsites) {   
        s = &sites[siteidx];
        siteidx += 1;
        return s;
    } else    
        return ((Site *) NULL);
}

void makefree(Freenode *curr, Freelist *fl)
{
    curr->nextfree = fl->head;
    fl->head = curr;
}

gchar* getfree(Freelist *fl)
{
    gint i; 
    Freenode *t;

    if (fl->head == (Freenode *) NULL) {   
        t = (Freenode *) myalloc(sqrt_nsites * fl->nodesize);

        if (t == 0)
            return 0;
        
        currentMemoryBlock->next = g_new(FreeNodeArrayList, 1);
        currentMemoryBlock = currentMemoryBlock->next;
        currentMemoryBlock->memory = t;
        currentMemoryBlock->next = NULL;

        for (i = 0; i < sqrt_nsites; i++)   
            makefree((Freenode *)((gchar *)t + i*fl->nodesize), fl);        
    }
    t = fl->head;
    fl->head = (fl->head)->nextfree;
    return ((gchar *) t);
}

Halfedge* HEcreate(Edge *e, gint pm)
{
    Halfedge *answer;
    answer = (Halfedge *) getfree(&hfl);
    answer->ELedge = e;
    answer->ELpm = pm;
    answer->PQnext = (Halfedge *) NULL;
    answer->vertex = (Site *) NULL;
    answer->ELrefcnt = 0;
    return answer;
}

gboolean ELinitialize()
{
    gint i;
    freeinit(&hfl, sizeof **ELhash);
    ELhashsize = 2 * sqrt_nsites;
    ELhash = (Halfedge **) myalloc(sizeof *ELhash * ELhashsize);

    if(ELhash == NULL)
        return FALSE;

    for (i = 0; i < ELhashsize; i ++) 
        ELhash[i] = (Halfedge *) NULL;
    ELleftend = HEcreate((Edge *) NULL, 0);
    ELrightend = HEcreate((Edge *) NULL, 0);
    ELleftend->ELleft = (Halfedge *) NULL;
    ELleftend->ELright = ELrightend;
    ELrightend->ELleft = ELleftend;
    ELrightend->ELright = (Halfedge *) NULL;
    ELhash[0] = ELleftend;
    ELhash[ELhashsize - 1] = ELrightend;

    return TRUE;
}

gint PQempty()
{
    return (PQcount == 0);
}

Point PQ_min()
{
    Point answer;
    
    while (PQhash[PQmin].PQnext == (Halfedge *) NULL) PQmin += 1;
    answer.x = PQhash[PQmin].PQnext->vertex->coord.x;
    answer.y = PQhash[PQmin].PQnext->ystar;
    return answer;
}

Halfedge* ELgethash(gint b)
{
    Halfedge *he;
    
    if (b<0 || b>=ELhashsize) 
        return ((Halfedge *) NULL);
    he = ELhash[b]; 
    if (he == (Halfedge *) NULL || he->ELedge != (Edge *) DELETED) 
        return he;
    
    /* Hash table points to deleted half edge.  Patch as necessary. */
    ELhash[b] = (Halfedge *) NULL;
    if ((he -> ELrefcnt -= 1) == 0) 
        makefree((Freenode*) he, &hfl);
    return ((Halfedge *) NULL);
}

gint right_of(Halfedge *el, Point *p)
{
    Edge *e;
    Site *topsite;
    gint right_of_site, above, fast;
    gdouble dxp, dyp, dxs, t1, t2, t3, yl;
    
    e = el->ELedge;
    topsite = e->reg[1];
    right_of_site = p->x > topsite->coord.x;
    if (right_of_site && el->ELpm == le) return 1;
    if (!right_of_site && el->ELpm == re) return 0;
    
    if (e->a == 1.0) {   
        dyp = p->y - topsite->coord.y;
        dxp = p->x - topsite->coord.x;
        fast = 0;
        if ((!right_of_site & (e->b<0.0)) | (right_of_site & (e->b>=0.0)) ) {   
            above = dyp>= e->b*dxp; 
            fast = above;
        } else  {   
            above = p->x + p->y*e->b > e-> c;
            if(e->b<0.0) above = !above;
            if (!above) fast = 1;
        }
        if (!fast) {   
            dxs = topsite->coord.x - (e->reg[0])->coord.x;
            above = e->b * (dxp*dxp - dyp*dyp) <
            dxs*dyp*(1.0+2.0*dxp/dxs + e->b*e->b);
            if(e->b<0.0) above = !above;
        }
    } else  /*e->b==1.0 */ {   
        yl = e->c - e->a*p->x;
        t1 = p->y - yl;
        t2 = p->x - topsite->coord.x;
        t3 = yl - topsite->coord.y;
        above = t1*t1 > t2*t2 + t3*t3;
    }
    return (el->ELpm == le ? above : !above);
}

Halfedge* ELleftbnd(Point *p)
{
    gint i, bucket;
    Halfedge *he;
    
    /* Use hash table to get close to desired halfedge
     * use the hash function to find the place in the hash map 
     * that this HalfEdge should be */
    bucket = (gint)((p->x - xmin)/deltax * ELhashsize);  
    
    /* make sure that the bucket position in within the 
     * range of the hash array */
    if (bucket < 0) bucket = 0; 
    if (bucket >= ELhashsize) bucket = ELhashsize - 1;

    he = ELgethash(bucket);
    
    /* if the HE isn't found, search backwards and forwards 
     * in the hash map for the first non-null entry */
    if (he == (Halfedge *) NULL) {   
        for(i = 1; 1; i++) {   
            if ((he=ELgethash(bucket-i)) != (Halfedge *) NULL) 
                break;
            if ((he=ELgethash(bucket+i)) != (Halfedge *) NULL) 
                break;
        }
        totalsearch += i;
    }
    ntry += 1;
    /* Now search linear list of halfedges for the correct one */
    /* keep going right on the list until either the end is reached, 
     * or you find the 1st edge which the point */
    if (he == ELleftend || (he != ELrightend && right_of(he, p))) {
        do {
            he = he->ELright;
        } while (he != ELrightend && right_of(he, p)); 
        he = he -> ELleft;              /* isn't to the right of */
    /* if the point is to the left of the HalfEdge, then search left 
     * for the HE just to the left of the point */
    } else
        do  {
            he = he->ELleft;
        } while (he != ELleftend && !right_of(he, p));
        
    /* Update hash table and reference counts */
    if (bucket > 0 && bucket < ELhashsize-1) {   
        if (ELhash[bucket] != (Halfedge *) NULL)
            ELhash[bucket]->ELrefcnt -= 1;
        ELhash[bucket] = he;
        ELhash[bucket]->ELrefcnt += 1;
    };
    return he;
}

Halfedge* ELright(Halfedge *he)
{
    return (he->ELright);
}

Halfedge* ELleft(Halfedge *he)
{
    return (he->ELleft);
}

Site* rightreg(Halfedge *he)
{
    /* if this halfedge has no edge, return the bottom 
     * site (whatever that is) */
    if(he->ELedge == (Edge *) NULL) 
        return bottomsite;

    /* if the ELpm field is zero, return the site 0 that this edge bisects, 
     * otherwise return site number 1 */
    return (he->ELpm == le ? he->ELedge->reg[re] : he->ELedge->reg[le]);
}

void ref(Site *v)
{
    v->refcnt += 1;
}

void deref(Site *v)
{
    v->refcnt -= 1;
    if (v->refcnt == 0) 
        makefree((Freenode*) v, &sfl);
}

Edge* bisect(Site *s1, Site *s2)
{
    gdouble dx, dy, adx, ady;
    Edge *newedge;   

    newedge = (Edge *) getfree(&efl);
    
    newedge->reg[0] = s1; /* store the sites that this edge is bisecting */
    newedge->reg[1] = s2;
    ref(s1); 
    ref(s2);
    /* to begin with, there are no endpoints on the bisector - it goes 
     * to infinity */
    newedge->ep[0] = (Site *) NULL; 
    newedge->ep[1] = (Site *) NULL;
    
    /* get the difference in x dist between the sites */
    dx = s2->coord.x - s1->coord.x;  
    dy = s2->coord.y - s1->coord.y;
    /* make sure that the difference in positive */
    adx = dx>0 ? dx : -dx;                  
    ady = dy>0 ? dy : -dy;
    newedge -> c = (gdouble) (s1->coord.x * dx + s1->coord.y * dy + 
                              (dx*dx + dy*dy)*0.5); /* get the slope of line */

    if (adx > ady) {   
        /* set formula of line, with x fixed to 1 */
        newedge -> a = 1.0; newedge -> b = dy/dx; newedge -> c /= dx;
    } else {   
        /* set formula of line, with y fixed to 1 */
        newedge -> b = 1.0; newedge -> a = dx/dy; newedge -> c /= dy;
    }
    
    newedge->edgenbr = nedges;
    
    nedges += 1;
    return newedge;
}


void ELinsert(Halfedge *lb, Halfedge *newHe)
{
    newHe->ELleft = lb;
    newHe->ELright = lb->ELright;
    (lb->ELright)->ELleft = newHe;
    lb->ELright = newHe;
}

Site* intersect(Halfedge *el1, Halfedge *el2)
{
    Edge *e1;
    Edge *e2; 
    Edge *e;
    Halfedge *el;
    gdouble d, xint, yint;
    gint right_of_site;
    Site *v;
    
    e1 = el1->ELedge;
    e2 = el2->ELedge;
    if (e1 == (Edge*) NULL || e2 == (Edge*) NULL) 
        return ((Site *) NULL);

    /* if the two edges bisect the same parent, return null */
    if (e1->reg[1] == e2->reg[1])
        return ((Site *) NULL);
    
    d = e1->a * e2->b - e1->b * e2->a;
    if (-1.0e-10<d && d<1.0e-10) 
        return ((Site *) NULL);
    
    xint = (e1->c*e2->b - e2->c*e1->b)/d;
    yint = (e2->c*e1->a - e1->c*e2->a)/d;
    
    if( (e1->reg[1]->coord.y < e2->reg[1]->coord.y) ||
        (e1->reg[1]->coord.y == e2->reg[1]->coord.y &&
        e1->reg[1]->coord.x < e2->reg[1]->coord.x) ) {   
        el = el1; 
        e = e1;
    } else {   
        el = el2; 
        e = e2;
    }
    
    right_of_site = xint >= e -> reg[1] -> coord.x;
    if ((right_of_site && el -> ELpm == le) || 
        (!right_of_site && el -> ELpm == re)) 
        return ((Site *) NULL);
    
    /* create a new site at the point of intersection - this is a new 
     * vector event waiting to happen */
    v = (Site *) getfree(&sfl);
    v->refcnt = 0;
    v->coord.x = xint;
    v->coord.y = yint;
    return v;
}

gint PQbucket(Halfedge *he)
{
    gint bucket;
    
    bucket = (gint) ((he->ystar - ymin)/deltay * PQhashsize);
    if (bucket < 0) bucket = 0;
    if (bucket >= PQhashsize) bucket = PQhashsize - 1;
    if (bucket < PQmin) PQmin = bucket;
    return bucket;
}

void PQinsert(Halfedge *he, Site* v, gdouble offset)
{
    Halfedge *last; 
    Halfedge *next;
    
    he->vertex = v;
    ref(v);
    he->ystar = (gdouble) (v->coord.y + offset);
    last = &PQhash[PQbucket(he)];
    while ((next = last->PQnext) != (Halfedge *) NULL &&
           (he->ystar  > next->ystar  ||
            (he->ystar == next->ystar && v->coord.x > next->vertex->coord.x)))
        last = next;
    he->PQnext = last->PQnext; 
    last->PQnext = he;
    PQcount += 1;
}

void PQdelete(Halfedge *he)
{
    Halfedge *last;
    
    if (he->vertex != (Site *) NULL) {   
        last = &PQhash[PQbucket(he)];
        while (last->PQnext != he) 
            last = last -> PQnext;
        last->PQnext = he->PQnext;
        PQcount -= 1;
        deref(he->vertex);
        he->vertex = (Site *) NULL;
    }
}

gdouble dist(Site *s, Site *t)
{
    gdouble dx, dy;
    dx = s->coord.x - t->coord.x;
    dy = s->coord.y - t->coord.y;
    return (gdouble) (sqrt(dx*dx + dy*dy));
}

Halfedge* PQextractmin()
{
    Halfedge *curr;
    
    curr = PQhash[PQmin].PQnext;
    PQhash[PQmin].PQnext = curr->PQnext;
    PQcount -= 1;
    return curr;
}

Site* leftreg(Halfedge *he)
{
    if (he->ELedge == (Edge *) NULL) 
        return bottomsite;
    return (he -> ELpm == le ? he->ELedge->reg[le] : he->ELedge -> reg[re]);
}

void makevertex(Site *v)
{
    v->sitenbr = nvertices;
    nvertices += 1;
}

void pushGraphEdge(gdouble x1, gdouble y1, gdouble x2, gdouble y2)
{
    Graphedge* newEdge = g_new(Graphedge, 1);
    newEdge->next = allEdges;
    allEdges = newEdge;
    newEdge->x1 = x1;
    newEdge->y1 = y1;
    newEdge->x2 = x2;
    newEdge->y2 = y2;
}

void clip_line(Edge *e)
{
    Site *s1;
    Site *s2;
    gdouble x1 = 0; 
    gdouble x2 = 0;
    gdouble y1 = 0;
    gdouble y2 = 0;

    x1 = e->reg[0]->coord.x;
    x2 = e->reg[1]->coord.x;
    y1 = e->reg[0]->coord.y;
    y2 = e->reg[1]->coord.y;

    /* if the distance between the two points this line was created from 
     * is less than the square root of 2, then ignore it */
    if (sqrt(((x2 - x1) * (x2 - x1)) + ((y2 - y1) * (y2 - y1))) 
          < minDistanceBetweenSites)
        return;
    pxmin = borderMinX;
    pxmax = borderMaxX;
    pymin = borderMinY;
    pymax = borderMaxY;

    if (e->a == 1.0 && e->b >= 0.0) {   
        s1 = e->ep[1];
        s2 = e->ep[0];
    } else {
        s1 = e->ep[0];
        s2 = e->ep[1];
    }
    
    if(e -> a == 1.0) {
        y1 = pymin;
        if (s1 != (Site *) NULL && s1->coord.y > pymin)
            y1 = s1->coord.y;
        if (y1 > pymax) 
            y1 = pymax;
        x1 = e->c - e->b * y1;
        y2 = pymax;
        if (s2 != (Site *) NULL && s2->coord.y < pymax) 
            y2 = s2->coord.y;

        if (y2 < pymin) 
            y2 = pymin;
        x2 = (e->c) - (e->b) * y2;
        if (((x1 > pxmax) & (x2 > pxmax)) | ((x1 < pxmin) & (x2 < pxmin))) 
            return;
        if (x1 > pxmax) {
            x1 = pxmax; y1 = (e->c - x1)/e->b;
        }
        if (x1 < pxmin) {   
            x1 = pxmin; y1 = (e->c - x1)/e->b;
        }
        if (x2 > pxmax) {   
            x2 = pxmax; y2 = (e->c - x2)/e->b;
        }
        if (x2 < pxmin) {   
            x2 = pxmin; y2 = (e->c - x2)/e->b;
        }
    }
    else
    {
        x1 = pxmin;
        if (s1 != (Site *)NULL && s1->coord.x > pxmin) 
            x1 = s1->coord.x;
        if (x1>pxmax) 
            x1 = pxmax;
        y1 = e->c - e->a * x1;
        x2 = pxmax;
        if (s2 != (Site *)NULL && s2->coord.x < pxmax) 
            x2 = s2->coord.x;
        if (x2<pxmin) 
            x2 = pxmin;
        y2 = e->c - e->a * x2;
        if (((y1 > pymax) & (y2 > pymax)) | ((y1 < pymin) & (y2 < pymin))) 
            return;
        if (y1 > pymax) {   
            y1 = pymax; x1 = (e->c - y1)/e->a;
        }
        if (y1 < pymin) {   
            y1 = pymin; x1 = (e->c - y1)/e->a;
        }
        if (y2 > pymax) {   
            y2 = pymax; x2 = (e->c - y2)/e->a;
        }
        if (y2 < pymin) {   
            y2 = pymin; x2 = (e->c - y2)/e->a;
        }
    };
    
    pushGraphEdge(x1,y1,x2,y2);
}

void endpoint(Edge *e, gint lr, Site* s)
{
    e->ep[lr] = s;
    ref(s);
    if (e->ep[re-lr] == (Site *) NULL) 
        return;

    clip_line(e);

    deref(e->reg[le]);
    deref(e->reg[re]);
    makefree((Freenode*) e, &efl);
}

void ELdelete(Halfedge *he)
{
    (he->ELleft)->ELright = he->ELright;
    (he->ELright)->ELleft = he->ELleft;
    he->ELedge = (Edge *) DELETED;
}

gboolean voronoi(gint triangulate)
{
    Site *newsite;
    Site *bot;
    Site *top;
    Site *temp;
    Site *p;
    Site *v;
    Point newintstar;
    int pm;
    Halfedge *lbnd;
    Halfedge *rbnd;
    Halfedge *llbnd;
    Halfedge *rrbnd;
    Halfedge *bisector;
    Edge *e;
    
    PQinitialize();
    bottomsite = nextone();
    gboolean retval = ELinitialize();

    if (!retval)
        return FALSE;
    
    newsite = nextone();
    while (1) {

        if (!PQempty()) 
            newintstar = PQ_min();
        
        /* if the lowest site has a smaller y value than the lowest vector 
         * intersection, process the site otherwise process the vector 
         * intersection */

        if (newsite != (Site *) NULL  && 
            (PQempty() || newsite->coord.y < newintstar.y
             || (newsite->coord.y == newintstar.y && 
                 newsite->coord.x < newintstar.x))) {
            /* new site is smallest - this is a site event */
            lbnd = ELleftbnd(&(newsite->coord));/* get the first HalfEdge to 
                                                 * the LEFT of the new site */
            rbnd = ELright(lbnd);               /* get the first HalfEdge to 
                                                 * the RIGHT of the new site */
            bot = rightreg(lbnd);               /* if this halfedge has no edge
                                                 * bot = bottom site (whatever 
                                                 * that is) */
            e = bisect(bot, newsite);           /* create a new edge that 
                                                 * bisects */
            bisector = HEcreate(e, le);         /* create a new HalfEdge, 
                                                 * setting its ELpm field to 
                                                 * 0 */
            ELinsert(lbnd, bisector);           /* insert this new bisector 
                                                 * edge between the left and 
                                                 * right vectors in a linked 
                                                 * list */
            
            /* if the new bisector intersects with the left edge, remove the 
             * left edge's vertex, and put in the new one */
            if ((p = intersect(lbnd, bisector)) != (Site *) NULL) {   
                PQdelete(lbnd);
                PQinsert(lbnd, p, dist(p, newsite));
            }
            lbnd = bisector;          
            bisector = HEcreate(e, re);         /* create a new HalfEdge, 
                                                 * setting its ELpm field to 
                                                 * 1 */
            ELinsert(lbnd, bisector);           /* insert the new HE to the 
                                                 * right of the original 
                                                 * bisector earlier in the 
                                                 * IF stmt */
            
            /* if this new bisector intersects with the right edge
             * push the HE into the ordered linked list of vertices */
            if ((p = intersect(bisector, rbnd)) != (Site *) NULL)
                PQinsert(bisector, p, dist(p,newsite));
            newsite = nextone();
        /* intersection is smallest - this is a vector event */ 
        } else if (!PQempty()) {   
            lbnd = PQextractmin();                /* pop the HalfEdge with the 
                                                   * lowest vector off the 
                                                   * ordered list of vectors */
            llbnd = ELleft(lbnd);                 /* get the HalfEdge to the 
                                                   * left of the above HE */
            rbnd = ELright(lbnd);                 /* get the HalfEdge to the 
                                                   * right of the above HE */
            rrbnd = ELright(rbnd);                /* get the HalfEdge to the 
                                                   * right of the HE to the 
                                                   * right of the lowest HE */
            bot = leftreg(lbnd);                  /* get the Site to the left 
                                                   * of the left HE which it 
                                                   * bisects */
            top = rightreg(rbnd);                 /* get the Site to the right 
                                                   * of the right HE which it 
                                                   * bisects */
            v = lbnd->vertex;                     /* get the vertex that caused 
                                                   * this event */
            makevertex(v);                        /* set the vertex number - 
                                                   * couldn't do this earlier 
                                                   * since we didn't know when 
                                                   * it would be processed */
            endpoint(lbnd->ELedge, lbnd->ELpm, v);/* set the endpoint of the 
                                                   * left HalfEdge to be this 
                                                   * vector */
            endpoint(rbnd->ELedge, rbnd->ELpm, v);/* set the endpoint of the 
                                                   * right HalfEdge to be this 
                                                   * vector */
            ELdelete(lbnd);                       /* mark the lowest HE for 
                                                   * deletion - can't delete 
                                                   * yet because there might be 
                                                   * pointers to it in Hash 
                                                   * Map */
            PQdelete(rbnd);                       /* remove all vertex events 
                                                   * to do with the right HE */
            ELdelete(rbnd);                       /* mark the right HE for 
                                                   * deletion - can't delete 
                                                   * yet because there might be 
                                                   * pointers to it in Hash 
                                                   * Map */
            pm = le;                              /* set the pm variable to 
                                                   * zero */
            /* If the site to the left of the event is higher than the site to
             * the right of it, then swap them and set the pm variable to 1 */
            if (bot->coord.y > top->coord.y) {
                temp = bot; 
                bot = top; 
                top = temp; 
                pm = re;
            }
            e = bisect(bot, top);                 /* create an Edge (or line) 
                                                   * that is between the two 
                                                   * Sites. This creates
                                                   * the formula of the line, 
                                                   * and assigns a line number 
                                                   * to it */
            bisector = HEcreate(e, pm);           /* create a HE from the Edge 
                                                   * 'e', and make it point to 
                                                   * that edge with its ELedge 
                                                   * field */
            ELinsert(llbnd, bisector);            /* insert the new bisector to
                                                   * the right of left HE */
            endpoint(e, re-pm, v);                /* set one endpoint to the 
                                                   * new edge to be the vector 
                                                   * point 'v' */
                                                  /* If the site to the left of
                                                   * this bisector is higher 
                                                   * than the right site, then 
                                                   * this endpoint is put in 
                                                   * position 0; otherwise in 
                                                   * pos 1 */
            deref(v);                             /* delete the vector 'v' */

            /* if left HE and the new bisector don't intersect, then delete 
             * the left HE, and reinsert it */
            if ((p = intersect(llbnd, bisector)) != (Site *) NULL) {   
                PQdelete(llbnd);
                PQinsert(llbnd, p, dist(p, bot));
            }

            /* if right HE and the new bisector don't intersect, then 
             * reinsert it */
            if ((p = intersect(bisector, rrbnd)) != (Site *) NULL)
                PQinsert(bisector, p, dist(p,bot));
        } else break;
    }

    for(lbnd = ELright(ELleftend); lbnd != ELrightend; lbnd = ELright(lbnd)) {   
        e = lbnd -> ELedge;
        clip_line(e);
    }

    cleanup();

    return TRUE;
}

/*----------------------------------------------------------------------------- 
 *  voronoi_make  --  makes the Voronoi diagram given an array of points 
 *-----------------------------------------------------------------------------
 */
gboolean voronoi_make(gdouble *xValues, gdouble *yValues, gint numPoints, 
                      gdouble minX, gdouble maxX, gdouble minY, gdouble maxY, 
                      gdouble minDist)
{
    cleanup();
    cleanupEdges();
    gint i;

    minDistanceBetweenSites = minDist;

    nsites = numPoints;
    plot = 0;
    triangulate = 0;    
    debug = 1;
    sorted = 0; 
    freeinit(&sfl, sizeof(Site));
        
    sites = (Site *) myalloc(nsites * sizeof(*sites));

    if(sites == NULL)
        return FALSE;

    xmin = xValues[0];
    ymin = yValues[0];
    xmax = xValues[0];
    ymax = yValues[0];

    for (i = 0; i < nsites; i++) {
        sites[i].coord.x = xValues[i];
        sites[i].coord.y = yValues[i];
        sites[i].sitenbr = i;
        sites[i].refcnt = 0;

        if (xValues[i] < xmin)
            xmin = xValues[i];
        else if (xValues[i] > xmax)
            xmax = xValues[i];

        if (yValues[i] < ymin)
            ymin = yValues[i];
        else if (yValues[i] > ymax)
            ymax = yValues[i];
    }
    
    qsort(sites, nsites, sizeof(*sites), scomp);
    
    siteidx = 0;
    geominit();
    gdouble temp = 0;
    if (minX > maxX) {
        temp = minX;
        minX = maxX;
        maxX = temp;
    }
    if (minY > maxY) {
        temp = minY;
        minY = maxY;
        maxY = temp;
    }
    borderMinX = minX;
    borderMinY = minY;
    borderMaxX = maxX;
    borderMaxY = maxY;
    
    siteidx = 0;
    voronoi(triangulate);

    return TRUE;
}

/*----------------------------------------------------------------------------- 
 *  reset_iterator  --  resets the internal pointer to the head of the list
 *-----------------------------------------------------------------------------
 */
void reset_iterator()
{
    iteratorEdges = allEdges;
}

/*----------------------------------------------------------------------------- 
 *  get_next  --  gets the next edge, returns FALSE when it reaches the end
 *-----------------------------------------------------------------------------
 */
gboolean get_next(gdouble *x1, gdouble *y1, gdouble *x2, gdouble *y2)
{
    if (iteratorEdges == NULL)
        return FALSE;
    
    *x1 = iteratorEdges->x1;
    *x2 = iteratorEdges->x2;
    *y1 = iteratorEdges->y1;
    *y2 = iteratorEdges->y2;

    iteratorEdges = iteratorEdges->next;

    return TRUE;
}

/*----------------------------------------------------------------------------- 
 *  voronoi_create  --  creates carious structures and lists
 *-----------------------------------------------------------------------------
 */
void voronoi_create()
{
    siteidx = 0;
    sites = NULL;

    allMemoryList = g_new(FreeNodeArrayList, 1);
    allMemoryList->memory = NULL;
    allMemoryList->next = NULL;
    currentMemoryBlock = allMemoryList;
    allEdges = NULL;
    iteratorEdges = NULL;
    minDistanceBetweenSites = 0;
}

/*----------------------------------------------------------------------------- 
 *  voronoi_free  --  frees various structures and lists
 *-----------------------------------------------------------------------------
 */
void voronoi_free()
{
    cleanup();
    cleanupEdges();

    if (allMemoryList != 0)
        g_free(allMemoryList);
}

/*=============================================================================
 * END VORONOI CODE
 *===========================================================================*/

/*----------------------------------------------------------------------------- 
 *  remove_points  --  removes degenerate points from an array of points
 *-----------------------------------------------------------------------------
 */
void remove_points(GArray *apx, GArray *apy)
{
    gint n;
    gdouble x1, y1, x2, y2;
    for (n = apx->len - 1; n > 0; n--) {
        x1 = g_array_index(apx, gdouble, n-1);
        y1 = g_array_index(apy, gdouble, n-1);
        x2 = g_array_index(apx, gdouble, n);
        y2 = g_array_index(apy, gdouble, n);
        if ((x2 - x1)*(x2 - x1) + (y2 - y1)*(y2 - y1) < EPS*EPS) {
            g_array_remove_index(apx, n);
            g_array_remove_index(apy, n);
        }
    }
}

/*----------------------------------------------------------------------------- 
 *  get_stroke_points  --  gathers site data into arrays
 *-----------------------------------------------------------------------------
 */
void get_stroke_points(gint32 vectors_id, gint stroke_id,
                       GArray* apx, GArray* apy)
{
    gint      n, num_points;
    gboolean  closed;
    gdouble  *ctlpts;
    
    gimp_vectors_stroke_get_points(vectors_id, stroke_id, &num_points,
                                   &ctlpts, &closed);
    for (n = 2; n < num_points; n+= 6) {
        g_array_append_val(apx, ctlpts[n]);
        g_array_append_val(apy, ctlpts[n+1]);
    }
}

/*----------------------------------------------------------------------------- 
 *  voronoi_path  --  gathers data and then creates a Voronoi diagram path
 *-----------------------------------------------------------------------------
 */
void voronoi_path(gint32 image_id, gint32 vectors_id)
{
    gint32   new_vectors_id, new1_vectors_id;;
    gint     n, num_strokes;
    gint    *strokes;
    gchar   *v_name;
    gchar   *v1_name;
    GArray  *apx;
    GArray  *apy;
    gdouble *xp;
    gdouble *yp;
    gdouble  x1, y1, x2, y2;
    gdouble  ctlpts[12];
    
    /* We create a new vector and delete the old one (undo doesn't
     * work if you simply change the strokes of an existing vector) */
    strokes = gimp_vectors_get_strokes(vectors_id, &num_strokes);
    v_name = gimp_vectors_get_name(vectors_id);
    new_vectors_id = gimp_vectors_new(image_id, v_name);
    
    /* Arrays will hold x and y values of original points */
    apx = g_array_new(FALSE, TRUE, sizeof(gdouble));
    apy = g_array_new(FALSE, TRUE, sizeof(gdouble));
    
    /* Extract the points and remove duplicates */
    for (n = 0; n < num_strokes; n++) 
      get_stroke_points(vectors_id, strokes[n], apx, apy);
    remove_points(apx, apy);
      
    /* Work out Voronoi segments; each segment will be a new stroke */
    if (apx->len > 1) {
        xp = g_new(gdouble, apx->len);
        for (n = 0; n < apx->len; n++)
            xp[n] = g_array_index(apx, gdouble, n);
        yp = g_new(gdouble, apy->len);
        for (n = 0; n < apy->len; n++)
            yp[n] = g_array_index(apy, gdouble, n);
        voronoi_create();
        voronoi_make(xp, yp, apx->len, 0, gimp_image_width(image_id), 0, 
                     gimp_image_height(image_id), MIN_DIST);
        reset_iterator();
        while (get_next(&x1, &y1, &x2, &y2)) {
            ctlpts[0]  = x1; ctlpts[1]  = y1;
            ctlpts[2]  = x1; ctlpts[3]  = y1;
            ctlpts[4]  = x1; ctlpts[5]  = y1;
            ctlpts[6]  = x2; ctlpts[7]  = y2;
            ctlpts[8]  = x2; ctlpts[9]  = y2;
            ctlpts[10] = x2; ctlpts[11] = y2;
            gimp_vectors_stroke_new_from_points(new_vectors_id, 
                        GIMP_VECTORS_STROKE_TYPE_BEZIER, 12, ctlpts, FALSE);
        }
        voronoi_free();
        
        /* Create a new vector for the original points if necessary */
        if (cell_site) {
            if (g_str_has_suffix(v_name, "_original"))
                v1_name = g_strdup(v_name);
            else
                v1_name = g_strconcat(v_name, "_original", NULL);
            new1_vectors_id = gimp_vectors_new(image_id, v1_name);
            for (n = 0; n < apx->len; n++) {
                x1 = g_array_index(apx, gdouble, n);
                y1 = g_array_index(apy, gdouble, n);
                ctlpts[0]  = x1;     ctlpts[1]  = y1;
                ctlpts[2]  = x1;     ctlpts[3]  = y1;
                ctlpts[4]  = x1;     ctlpts[5]  = y1;
                ctlpts[6]  = x1+EXS; ctlpts[7]  = y1+EXS;
                ctlpts[8]  = x1+EXS; ctlpts[9]  = y1+EXS;
                ctlpts[10] = x1+EXS; ctlpts[11] = y1+EXS;
                gimp_vectors_stroke_new_from_points(new1_vectors_id, 
                        GIMP_VECTORS_STROKE_TYPE_BEZIER, 12, ctlpts, FALSE);
            }
        }
        
        /* Add the new vectors and remove the old */
        if (cell_site) {
            gimp_image_add_vectors(image_id, new1_vectors_id, 
                               gimp_image_get_vectors_position(image_id, 
                                                               vectors_id));
            gimp_image_add_vectors(image_id, new_vectors_id, 
                               gimp_image_get_vectors_position(image_id, 
                                                            new1_vectors_id));
        } else
            gimp_image_add_vectors(image_id, new_vectors_id, 
                               gimp_image_get_vectors_position(image_id, 
                                                               vectors_id));
        
        gimp_image_remove_vectors(image_id, vectors_id);
        gimp_vectors_set_name(new_vectors_id, v_name);
        
        g_free(xp);
        g_free(yp);
    }
    g_array_free(apx, TRUE);
    g_array_free(apy, TRUE);
    g_free(v_name);
    g_free(strokes);
}

/*----------------------------------------------------------------------------- 
 *  voronoi_dialog  --  dialog that allows user to set algorithm parameters
 *-----------------------------------------------------------------------------
 */
gboolean voronoi_dialog(void)
{
    GtkWidget *dialog;
    GtkWidget *vbox;
    GtkWidget *toggle;
    gboolean   run;
    
    gimp_ui_init (PLUG_IN_BINARY, FALSE);
    
    dialog = gimp_dialog_new("Voronoi Path", PLUG_IN_BINARY,
                             NULL, 0,
                             gimp_standard_help_func, PLUG_IN_PROC,
                             GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                             GTK_STOCK_OK,     GTK_RESPONSE_OK,
                             NULL);
                             
    gtk_dialog_set_alternative_button_order(GTK_DIALOG (dialog),
                                            GTK_RESPONSE_OK,
                                            GTK_RESPONSE_CANCEL,
                                            -1);
                                            
    gtk_window_set_resizable (GTK_WINDOW (dialog), FALSE);
    
    vbox = gtk_vbox_new(FALSE, 12);
    gtk_container_set_border_width(GTK_CONTAINER(vbox), 12);
    gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), vbox);
    gtk_widget_show(vbox);
    
    toggle 
      = gtk_check_button_new_with_mnemonic("_Preserve original path");
    gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);
    gtk_widget_show(toggle);
    g_signal_connect(toggle, "toggled",
                     G_CALLBACK(gimp_toggle_button_update),
                     &cell_site);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(toggle),
                                 (cell_site == TRUE));
                     
    gtk_widget_show(dialog);
    
    run = (gimp_dialog_run(GIMP_DIALOG(dialog)) == GTK_RESPONSE_OK);

    gtk_widget_destroy(dialog);

    return run;
}

/*----------------------------------------------------------------------------- 
 *  run  --  code that gets called when the plugin is asked to run
 *-----------------------------------------------------------------------------
 */
static void run(const gchar      *name,
                gint              nparams,
                const GimpParam  *param,
                gint             *nreturn_vals,
                GimpParam       **return_vals)
{
    static GimpParam  values[1];
    GimpPDBStatusType status = GIMP_PDB_SUCCESS;
    GimpRunMode       run_mode;
    gint32            vectors_id;
    gint32            image_id;

    /* Setting mandatory output values */
    *nreturn_vals = 1;
    *return_vals  = values;
    values[0].type = GIMP_PDB_STATUS;
    values[0].data.d_status = status;
    
    /* Extract various element ids */
    run_mode   = param[0].data.d_int32;
    image_id   = param[1].data.d_image;
    vectors_id = param[2].data.d_int32;
    
    switch (run_mode) {
        case GIMP_RUN_INTERACTIVE:
            /* Get options last values if needed */
            gimp_get_data(PLUG_IN_PROC, &cell_site);
            /* Display the dialog */
            if (!voronoi_dialog())
                return;
            break;
        case GIMP_RUN_NONINTERACTIVE:
            if (nparams != 4)
                status = GIMP_PDB_CALLING_ERROR;
            if (status == GIMP_PDB_SUCCESS) {
                cell_site = param[3].data.d_int32;
            }
            break;
        case GIMP_RUN_WITH_LAST_VALS:
            /*  Get options last values if needed  */
            status = GIMP_PDB_PASS_THROUGH;
            break;
        default:
            break;
    }
    
    if (status == GIMP_PDB_SUCCESS) {
        /* Bundle the smooth_path code inside an undo group */        
        gimp_image_undo_group_start(image_id);
        voronoi_path(image_id, vectors_id);
        gimp_image_undo_group_end(image_id);
      
        /* Refresh and clean up */
        if (run_mode != GIMP_RUN_NONINTERACTIVE)
            gimp_displays_flush();

        /*  Finally, set options in the core  */
        if (run_mode == GIMP_RUN_INTERACTIVE)
              gimp_set_data(PLUG_IN_PROC, &cell_site, sizeof(gint32));
    }
      
    values[0].data.d_status = status;
}
